<?php
// Start session for error handling
session_start();

// Check if form was submitted via POST
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    
    // Sanitize and validate input data
    $name = isset($_POST['name']) ? trim($_POST['name']) : '';
    $email = isset($_POST['email']) ? trim($_POST['email']) : '';
    $phone = isset($_POST['phone']) ? trim($_POST['phone']) : '';
    $service = isset($_POST['service']) ? trim($_POST['service']) : '';
    $message = isset($_POST['message']) ? trim($_POST['message']) : '';
    
    // Validation errors array
    $errors = [];
    
    // Validate Name
    if (empty($name)) {
        $errors[] = "Name is required.";
    } elseif (strlen($name) < 2) {
        $errors[] = "Name must be at least 2 characters long.";
    } elseif (!preg_match("/^[a-zA-Z\s'-]+$/", $name)) {
        $errors[] = "Name contains invalid characters.";
    }
    
    // Validate Email
    if (empty($email)) {
        $errors[] = "Email is required.";
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = "Invalid email format.";
    }
    
    // Validate Service
    if (empty($service)) {
        $errors[] = "Service type is required.";
    }
    
    // Validate Message
    if (empty($message)) {
        $errors[] = "Message is required.";
    } elseif (strlen($message) < 10) {
        $errors[] = "Message must be at least 10 characters long.";
    } elseif (strlen($message) > 1000) {
        $errors[] = "Message must not exceed 1000 characters.";
    }
    
    // If there are validation errors, redirect back with error
    if (!empty($errors)) {
        $_SESSION['form_errors'] = $errors;
        header("Location: contact.php?error=1");
        exit();
    }
    
    // Sanitize data for security
    $name = htmlspecialchars($name, ENT_QUOTES, 'UTF-8');
    $email = filter_var($email, FILTER_SANITIZE_EMAIL);
    $phone = htmlspecialchars($phone, ENT_QUOTES, 'UTF-8');
    $service = htmlspecialchars($service, ENT_QUOTES, 'UTF-8');
    $message = htmlspecialchars($message, ENT_QUOTES, 'UTF-8');
    
    // Email configuration (you can customize this)
    $to = "hello@sagestrategiesgroup.org"; // Change this to your actual email
    $subject = "New Contact Form Submission - " . $service;
    
    // Email body
    $email_body = "New contact form submission from SAGE STRATEGIES website:\n\n";
    $email_body .= "Name: " . $name . "\n";
    $email_body .= "Email: " . $email . "\n";
    if (!empty($phone)) {
        $email_body .= "Phone: " . $phone . "\n";
    }
    $email_body .= "Service Type: " . $service . "\n";
    $email_body .= "Message:\n" . $message . "\n";
    $email_body .= "\n---\n";
    $email_body .= "Submitted on: " . date('Y-m-d H:i:s') . "\n";
    
    // Email headers
    $headers = "From: " . $email . "\r\n";
    $headers .= "Reply-To: " . $email . "\r\n";
    $headers .= "X-Mailer: PHP/" . phpversion();
    
    // Attempt to send email
    // Note: For production, you may want to use a more robust email solution like PHPMailer
    $mail_sent = mail($to, $subject, $email_body, $headers);
    
    // Optional: Save to database or file (uncomment and configure as needed)
    /*
    // Example: Save to a text file
    $log_entry = date('Y-m-d H:i:s') . " | " . $name . " | " . $email . " | " . $service . " | " . $message . "\n";
    file_put_contents('contact_submissions.txt', $log_entry, FILE_APPEND);
    */
    
    // Redirect based on result
    if ($mail_sent) {
        // Success - redirect to contact page with success message
        header("Location: contact.php?success=1");
        exit();
    } else {
        // Failure - redirect with error
        $_SESSION['form_errors'] = ["Failed to send email. Please try again later."];
        header("Location: contact.php?error=1");
        exit();
    }
    
} else {
    // If accessed directly without POST, redirect to contact page
    header("Location: contact.php");
    exit();
}
?>

